#!/bin/sh
# Copyright (c) 2004-2015 Dmitry V. Levin <ldv@altlinux.org>
# Copyright (c) 2015-2017 The strace developers.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

set -efu

me="${0##*/}"
mydir="${0%/*}"
msg()
{
	printf >&2 '%s\n' "$me: $*"
}

case $# in
	1)
		inc_dir="$1"
		arch_dir=
		;;
	2)
		inc_dir="$1"
		arch_dir="$2"
		;;
	*)
	        echo >&2 "usage: $me include-directory [arch-include-directory]"
	        exit 1
		;;
esac

# Check and canonicalize include-directory and arch-include-directory.
abs_inc_dir="$(cd "$inc_dir" && pwd -P)"
INCLUDES_inc="-I$abs_inc_dir/uapi -I$abs_inc_dir"
abs_arch_dir=
INCLUDES_arch=
[ -z "$arch_dir" ] || {
	abs_arch_dir="$(cd "$arch_dir" && pwd -P)"
	INCLUDES_arch="-I$abs_arch_dir/uapi -I$abs_arch_dir"
}

cleanup()
{
	trap - EXIT
	rm -f ioctls_hex.h ioctls_sym.h
	exit "$@"
}
trap 'cleanup $?' EXIT
trap 'cleanup 1' HUP PIPE INT QUIT TERM

# Fetch ioctl commands defined in hex form.
{
	"$mydir"/ioctls_hex.sh "$inc_dir" 03 linux/hdreg.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 22 scsi/sg.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 46 linux/fb.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 4B linux/kd.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 4C linux/loop.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 53 linux/cdrom.h scsi/scsi.h scsi/scsi_ioctl.h
	"$mydir"/ioctls_hex.sh "$inc_dir" '\(46\|54\|66\|74\)' asm/ioctls.h asm-generic/ioctls.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 56 linux/vt.h
	"$mydir"/ioctls_hex.sh "$inc_dir" '7[12]' linux/videotext.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 89 asm/sockios.h asm-generic/sockios.h linux/sockios.h
	"$mydir"/ioctls_hex.sh "$inc_dir" 8B linux/wireless.h
} > ioctls_hex.h
msg "generated $(grep -c '^{' ioctls_hex.h) hex ioctls from $inc_dir"

# Fetch ioctl commands defined in symbolic form.
INCLUDES="$INCLUDES_arch ${INCLUDES-}" \
	"$mydir"/ioctls_sym.sh "$inc_dir" > ioctls_sym.h

# Move KVM_* constants from ioctls_inc.h to ioctls_arch.h.
grep -F 'linux/kvm.h' < ioctls_sym.h > ioctls_kvm.h ||:
grep -Fv 'linux/kvm.h' < ioctls_sym.h > ioctls_unkvm.h
mv ioctls_unkvm.h ioctls_sym.h

# Part of android ioctl commands are defined elsewhere.
android_dir="$inc_dir/../drivers/staging/android"
if [ -d "$android_dir/uapi" ]; then
	INCLUDES="$INCLUDES_inc $INCLUDES_arch ${INCLUDES-}" \
	"$mydir"/ioctls_sym.sh "$android_dir" staging/android >> ioctls_sym.h
fi
msg "generated $(grep -c '^{' ioctls_sym.h) symbolic ioctls from $inc_dir"

# Output all ioctl definitions fetched from include-directory.
echo "/* Generated by $me from definitions found in ${inc_dir%%/}/ tree. */" > ioctls_inc.h
LC_COLLATE=C sort -u ioctls_hex.h ioctls_sym.h >> ioctls_inc.h
msg "generated $(grep -c '^{' ioctls_inc.h) ioctls from $inc_dir"

[ -n "$arch_dir" ] || exit 0

# Fetch ioctl commands defined in hex form.
{
	"$mydir"/ioctls_hex.sh "$arch_dir" 54 asm/ioctls.h
	"$mydir"/ioctls_hex.sh "$arch_dir" '\(46\|54\|66\|74\)' asm/ioctls.h
	"$mydir"/ioctls_hex.sh "$arch_dir" 89 asm/sockios.h
} > ioctls_hex.h
msg "generated $(grep -c '^{' ioctls_hex.h) hex ioctls from $arch_dir"

# Fetch ioctl commands defined in symbolic form.
INCLUDES="$INCLUDES_inc ${INCLUDES-}" \
	"$mydir"/ioctls_sym.sh "$arch_dir" > ioctls_sym.h
msg "generated $(grep -c '^{' ioctls_sym.h) symbolic ioctls from $arch_dir"

# Output all ioctl definitions fetched from arch-include-directory.
echo "/* Generated by $me from definitions found in ${arch_dir%%/}/ tree. */" > ioctls_arch.h
LC_COLLATE=C sort -u ioctls_hex.h ioctls_kvm.h ioctls_sym.h >> ioctls_arch.h
msg "generated $(grep -c '^{' ioctls_arch.h) ioctls from $arch_dir"
